﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace WorldLab.MathLib
{
    /// <summary>
    /// matrix
    /// </summary>
    public class Matrix
    {
        private int numColumns = 0;/// matrix columns
		private int	numRows = 0;/// matrix rows
		private double eps = 0.0;
		private double[] elements = null;

        public int Columns/// matrix columns
		{
			get
			{
				return numColumns;
			}
		}
        public int Rows/// matrix rows
		{
			get
			{
                
				return numRows;
			}
		}
		public double Eps
		{
			get
			{
				return eps;
			}
			set
			{
				eps = value;
			}
		}
        public double this[int row, int col]
        {
            get
            {
                return elements[col + row * numColumns];
            }
            set
            {
                elements[col + row * numColumns] = value;
            }
        }

        public Matrix()
		{
			numColumns = 1;
			numRows = 1;
			Init(numRows, numColumns);
		}

		public Matrix(int nRows, int nCols)
		{
			numRows = nRows;
			numColumns = nCols;
			Init(numRows, numColumns);
		}

        public Matrix(double[,] value)
		{
			numRows = value.GetLength(0);
			numColumns = value.GetLength(1);
			double[] data = new double[numRows * numColumns];
			int k = 0;
			for (int i=0; i<numRows; ++i)
			{
				for (int j=0; j<numColumns; ++j)
				{
					data[k++] = value[i, j]; 
				}
			}
			Init(numRows, numColumns);
			SetData(data);
		}

        public Matrix(int nRows, int nCols, double[] value)
		{
			numRows = nRows;
			numColumns = nCols;
			Init(numRows, numColumns);
			SetData(value);
		}

        public Matrix(int nSize)
		{
			numRows = nSize;
			numColumns = nSize;
			Init(nSize, nSize);
		}

        public Matrix(int nSize, double[] value)
		{
			numRows = nSize;
			numColumns = nSize;
			Init(nSize, nSize);
			SetData(value);
		}

        public Matrix( Matrix source)
		{
            numColumns = source.GetNumColumns();
            numRows = source.GetNumRows();
			Init(numRows, numColumns);
            SetData(source.elements);
		}
        
        public bool Init(int nRows, int nCols)
		{
			numRows = nRows;
			numColumns = nCols;
			int nSize = nCols * nRows;
            if (nSize <= 0)
            {
                return false;
            }
			elements = new double[nSize];
		
			return true;
		}

        public bool MakeUnitMatrix(int nSize)
        {
            if (!Init(nSize, nSize))
            {
                return false;
            }

            for (int i = 0; i < nSize; ++i)
            {
                for (int j = 0; j < nSize; ++j)
                {
                    if (i == j)
                    {
                        SetElement(i, j, 1);
                    }
                }
            }

            return true;
        }

        public void SetData(double[] value)
        {
            elements = (double[])value.Clone();
        }

        public bool SetElement(int nRow, int nCol, double value)
        {
            if (nRow < 0 || nRow >= numRows || nCol < 0 || nCol >= numColumns)
            {
                return false;
            }

            elements[nCol + nRow * numColumns] = value;

            return true;
        }

        public void SetValue(Matrix source)
        {
            if (source != this)
            {
                Init(source.GetNumRows(), source.GetNumColumns());
                SetData(source.elements);
            }
        }

        public void SetRowVector(int nRow, double[] pVector)
        {
            for (int j = 0; j < numColumns; ++j)
            {
                SetElement(nRow, j, pVector[j]);
            }
        }

        public void SetColVector(int nCol, double[] pVector)
        {
            for (int i = 0; i < numRows; ++i)
            {
                SetElement(i, nCol, pVector[i]);
            }
        }

        public void SetEps(double newEps)
        {
            eps = newEps;
        }

        public double GetElement(int nRow, int nCol)
        {
            return elements[nCol + nRow * numColumns];
        }

        public int GetNumColumns()
        {
            return numColumns;
        }

        public int GetNumRows()
        {
            return numRows;
        }

        public double[] GetData()
        {
            return elements;
        }

        public double[] GetRowVector(int nRow)
        {
            double[] result = new double[numColumns];
            for (int j = 0; j < numColumns; ++j)
            {
                result[j] = GetElement(nRow, j);
            }

            return result;
        }

        public double[] GetColVector(int nCol)
        {
            double[] result = new double[numRows];
            for (int i = 0; i < numRows; ++i)
            {
                result[i] = GetElement(i, nCol);
            }

            return result;
        }

        public double GetEps()
        {
            return eps;
        }

        public static Matrix operator +(Matrix m1, Matrix m2)
		{
			return m1.Add(m2);
		}

        public static Matrix operator -(Matrix m1, Matrix m2)
		{
			return m1.Subtract(m2);
		}

        public static Matrix operator *(Matrix m1, Matrix m2)
		{
			return m1.Multiply(m2);
		}

        public static implicit operator double[](Matrix m)
		{
			return m.elements;
		}

        /// Matrix Addition
        public Matrix Add(Matrix source)
        {
            if (numColumns != source.GetNumColumns() || numRows != source.GetNumRows())
            {
                throw new Exception("The row/column number of the matrix does not match.");
            }

            Matrix result = new Matrix(this);

            for (int i = 0; i < numRows; ++i)
            {
                for (int j = 0; j < numColumns; ++j)
                {
                    result.SetElement(i, j, result.GetElement(i, j) + source.GetElement(i, j));
                }
            }

            return result;
        }


        /// Matrix Subtraction
        public Matrix Subtract(Matrix source)
        {
            if (numColumns != source.GetNumColumns() || numRows != source.GetNumRows())
            {
                throw new Exception("The row/column number of the matrix does not match.");
            }

            Matrix result = new Matrix(this);

            for (int i = 0; i < numRows; ++i)
            {
                for (int j = 0; j < numColumns; ++j)
                {
                    result.SetElement(i, j, result.GetElement(i, j) - source.GetElement(i, j));
                }
            }

            return result;
        }

        /// scalar multiplication of matrix
        public Matrix Multiply(double value)
        {
            Matrix result = new Matrix(this);

            for (int i = 0; i < numRows; ++i)
            {
                for (int j = 0; j < numColumns; ++j)
                {
                    result.SetElement(i, j, result.GetElement(i, j) * value);
                }
            }

            return result;
        }

        public Vector Multiply(Vector vector)
        {
            if (numColumns != vector.GetNums())
            {
                throw new Exception("Matrixes does not match.");
            }

            Vector result = new Vector(numRows);
  
            // [A][B][C]   [G]     [A*G + B*H + C*I]
            // [D][E][F] * [H] =   [D*G + E*H + F*I]
            //             [I]
            double value;
            for (int i = 0; i < result.GetNums(); ++i)
            {
                value = 0.0;
                for (int j = 0; j < numColumns; ++j)
                {
                    value += GetElement(i, j) * vector.GetElement(j);
                }
                result.SetElement(i, value);
            }

            return result;
        }

        public Matrix Multiply(Matrix source)
        {
            if (numColumns != source.GetNumRows())
            {
                throw new Exception("Matrixes does not match.");
            }

            Matrix result = new Matrix(numRows, source.GetNumColumns());

            // [A][B][C]   [G][H]     [A*G + B*I + C*K][A*H + B*J + C*L]
            // [D][E][F] * [I][J] =   [D*G + E*I + F*K][D*H + E*J + F*L]
            //             [K][L]
            double value;
            for (int i = 0; i < result.GetNumRows(); ++i)
            {
                for (int j = 0; j < source.GetNumColumns(); ++j)
                {
                    value = 0.0;
                    for (int k = 0; k < numColumns; ++k)
                    {
                        value += GetElement(i, k) * source.GetElement(k, j);
                    }

                    result.SetElement(i, j, value);
                }
            }

            return result;
        }

        public Matrix Transpose()
        {
            Matrix result = new Matrix(numColumns, numRows);

            for (int i = 0; i < numColumns; ++i)
            {
                for (int j = 0; j < numRows; ++j)
                {
                    result.SetElement(i, j, GetElement(j, i));
                }
            }

            return result;
        }

        public void SwapRow(int nRow1, int nRow2)
        {
            double[] pVector1 = GetRowVector(nRow1);
            double[] pVector2 = GetRowVector(nRow2);
            SetRowVector(nRow1, pVector2);
            SetRowVector(nRow2, pVector1);
            pVector1 = null;
            pVector2 = null;
        }

        public void SwapCol(int nCol1, int nCol2)
        {
            double[] pVector1 = GetRowVector(nCol1);
            double[] pVector2 = GetRowVector(nCol2);
            SetRowVector(nCol1, pVector2);
            SetRowVector(nCol2, pVector1);
            pVector1 = null;
            pVector2 = null;
        }

        public override bool Equals(object source)
        {
            Matrix matrix = source as Matrix;
            if (matrix == null)
            {
                return false;
            }

            if (numColumns != matrix.GetNumColumns() || numRows != matrix.GetNumRows())
            {
                return false;
            }

            for (int i = 0; i < numRows; ++i)
            {
                for (int j = 0; j < numColumns; ++j)
                {
                    if (Math.Abs(GetElement(i, j) - matrix.GetElement(i, j)) > eps)
                    {
                        return false;
                    }
                }
            }

            return true;
        }

        public override int GetHashCode()
        {
            double sum = 0;
            for (int i = 0; i < numRows; ++i)
            {
                for (int j = 0; j < numColumns; ++j)
                {
                    sum += Math.Abs(GetElement(i, j));
                }
            }
            return (int)Math.Sqrt(sum);
        }

        /// change to array
        public double[][] ToArray()
        {
            double[][] result = new double[numRows][];
            for (int i = 0; i < numRows; ++i)
            {
                result[i] = new double[numColumns];
            }

            for (int i = 0; i < numRows; ++i)
            {
                for (int j = 0; j < numRows; ++j)
                {
                    result[i][j] = GetElement(i, j);
                }
            }
            return result;
        }

        /// duplicat
        public Matrix Duplicate()
        {
            return new Matrix(this);
        }

        /// compute inverse matrix
        public Matrix Invert()
        {
            double[][] tt = ToArray();
            int n = tt.Length;

            Matrix result = Duplicate();

            int[] perm;
            int toggle;
            Matrix lum = Decompose(tt, out perm, out toggle);
            if (lum == null)
                return null;

            double[] b = new double[n];
            for (int i = 0; i < n; ++i)
            {
                for (int j = 0; j < n; ++j)
                {
                    if (i == perm[j])
                        b[j] = 1.0;
                    else
                        b[j] = 0.0;
                }

                double[] x = HelperSolve(lum, b);

                for (int j = 0; j < n; ++j)
                {
                    result.SetElement(j, i, x[j]);
                }
            }
            return result;
        }

        /// LU decompose
        public Matrix Decompose(double[][] matrix, out int[] perm, out int tog)
        {
            int rows = matrix.Length;
            int cols = matrix[0].Length;
            if (rows != cols)
                throw new Exception("Not a square matrix!");

            int n = rows;

            Matrix result = Duplicate();

            perm = new int[n];
            for (int i = 0; i < n; ++i)
            {
                perm[i] = i;
            }

            tog = 1;

            double ajj, aij;

            for (int j = 0; j < n - 1; ++j)
            {
                double max = Math.Abs(result.GetElement(j, j));
                int pRow = j;
                for (int i = j + 1; i < n; ++i)
                {
                    aij = Math.Abs(result.GetElement(i, j));
                    if (aij > max)
                    {
                        max = aij;
                        pRow = i;
                    }
                }

                if (pRow != j)
                {
                    result.SwapRow(pRow, j);

                    int tmp = perm[pRow];
                    perm[pRow] = perm[j];
                    perm[j] = tmp;

                    tog = -tog;
                }

                ajj = result.GetElement(j, j);
                if (Math.Abs(ajj) < 0.00000001)
                    return null;

                for (int i = j + 1; i < n; ++i)
                {
                    aij = result.GetElement(i, j) / ajj;
                    result.SetElement(i, j, aij);
                    for (int k = j + 1; k < n; ++k)
                    {
                        result.SetElement(i, k, result.GetElement(i, k) - aij * result.GetElement(j, k));
                    }
                }
            }

            return result;
        }

        static double[] HelperSolve(Matrix luMatrix, double[] b)
        {
            int n = luMatrix.GetNumRows();

            double[] x = new double[n];
            b.CopyTo(x, 0);

            // Ly = b
            for (int i = 1; i < n; ++i)
            {
                double sum = x[i];
                for (int j = 0; j < i; ++j)
                {
                    sum -= luMatrix.GetElement(i, j) * x[j];
                }
                x[i] = sum;
            }

            // Ux = y
            x[n - 1] /= luMatrix.GetElement(n - 1, n - 1);
            for (int i = n - 2; i >= 0; --i)
            {
                double sum = x[i];
                for (int j = i + 1; j < n; ++j)
                {
                    sum -= luMatrix.GetElement(i, j) * x[j];
                }
                x[i] = sum / luMatrix.GetElement(i, i);
            }

            return x;
        }

    }
}

